// === ERC View 수정본 (Download Audio 버튼 및 버튼 크기 통일 적용) ===
import { sendToWhisper, evaluateInterviewAnswer } from '../api.js';
import { setupRecording } from '../recorder.js';

let presentationText = '';
let countdownTimer = null;
const MAX_COUNTDOWN_TIME = 300;

export function renderERCView() {
  const ercContainer = document.getElementById('ercContent');
  ercContainer.innerHTML = `
    <div>
      <p><strong>🎯 Topic:</strong> Summarize and interpret your written composition</p>
      <div class="controls-row">
        <button id="record" class="green">🎙️ Start Recording</button>
        <button id="stop" class="red" disabled>⏹️ Stop</button>
      </div>
      <div class="controls-row">
        <button id="submit" class="orange" style="min-width:120px;">📤 Submit for Evaluation</button>
        <button id="downloadFeedbackBtn" class="orange" disabled style="min-width:120px;">📁 Download Feedback</button>
        <button id="reset" class="orange" style="min-width:120px;">🔄 Reset</button>
      </div>

      <p id="recordTimerDisplay" style="font-size: 13px; color: #888;"></p>
      <audio id="audio" controls></audio>
      <div class="controls-row">
        <button id="downloadAudioBtn" class="orange" style="display: none;">🎧 Download Audio</button>
      </div>
      <a id="downloadLink" style="display: none;" download></a>

      <h4>📝 ERC Presentation Transcript</h4>
      <div id="ercHistory" style="background:#f9f9f9; padding:10px; border:1px solid #ccc; border-radius:6px; min-height: 100px;"></div>
      <h4>📊 Academic Evaluation Feedback</h4>
      <pre id="ercFeedback"></pre>
    </div>
  `;

  const recordBtn = document.getElementById('record');
  const stopBtn = document.getElementById('stop');
  const submitBtn = document.getElementById('submit');
  const resetBtn = document.getElementById('reset');
  const feedbackBox = document.getElementById('ercFeedback');
  const historyBox = document.getElementById('ercHistory');
  const downloadBtn = document.getElementById('downloadFeedbackBtn');
  const audioDownloadBtn = document.getElementById('downloadAudioBtn');
  const downloadLink = document.getElementById('downloadLink');

  downloadLink.setAttribute('download', 'ERC_Audio.webm');

  setupRecording(recordBtn, stopBtn, document.getElementById('audio'), downloadLink, (blob) => {
    const audioUrl = URL.createObjectURL(blob);
    downloadLink.href = audioUrl;
    audioDownloadBtn.style.display = 'inline-block';
    audioDownloadBtn.onclick = () => downloadLink.click();
  });

  window.startCountdownTimer = function () {
    let remaining = MAX_COUNTDOWN_TIME;
    document.getElementById('recordTimerDisplay').textContent = `Time left: ${remaining}s`;

    countdownTimer = setInterval(() => {
      remaining--;
      document.getElementById('recordTimerDisplay').textContent = `Time left: ${remaining}s`;
      if (remaining <= 0) {
        clearInterval(countdownTimer);
        countdownTimer = null;
        stopBtn.click();
        alert("Time's up! Recording stopped automatically.");
      }
    }, 1000);
  };

  window.stopCountdownTimer = function () {
    if (countdownTimer) clearInterval(countdownTimer);
    countdownTimer = null;
    document.getElementById('recordTimerDisplay').textContent = '';
  };

  recordBtn.addEventListener('click', () => {
    window.startCountdownTimer();
  });

  stopBtn.addEventListener('click', window.stopCountdownTimer);

submitBtn.addEventListener('click', async () => {
  const blob = window.latestRecordingBlob;
  if (!blob) return alert("Please record your presentation first.");

  const transcript = await sendToWhisper(blob);
  presentationText = transcript;

  // 🚫 악의적 GPT 관련 명령어 차단
  const bannedPhrases = [
    "prompt", "chatgpt", "gpt", "openai", "api key", "system prompt",
    "write me", "generate", "make a", "please write", "how do i", "code"
  ];
  const lower = transcript.toLowerCase();
  const isAbuse = bannedPhrases.some(p => lower.includes(p));

  if (isAbuse) {
    alert("⚠️ 시스템 관련 명령어나 GPT 프롬프트 조작 시도가 탐지되었습니다. 작성한 글에 대해 자연스럽게 설명해 주세요.");
    return;
  }

  const bubble = document.createElement('div');
  bubble.className = 'bubble';
  bubble.style.cssText = 'background:#dcf8c6; padding:10px; border-radius:10px; margin-bottom:8px';
  bubble.textContent = transcript;
  historyBox.innerHTML = '';
  historyBox.appendChild(bubble);

  const evaluationPrompt = `
You are evaluating a student's oral presentation based on their own written composition.

The student is expected to clearly summarize the key points, provide insightful interpretation, and communicate with academic fluency.

Assess the presentation using these five criteria:

1. Clarity & Organization – Is the summary logically structured and easy to follow?
2. Interpretation & Insight – Does the student demonstrate critical thinking and go beyond surface-level summary?
3. Understanding of Source – Are the original ideas accurately conveyed and analyzed?
4. Language & Academic Style – Is the English appropriate for an academic setting?
5. Delivery & Confidence – Is the speaker articulate, fluent, and professional?

Provide a score out of 5. Include strengths and specific improvement suggestions.`;

  const feedback = await evaluateInterviewAnswer(`${evaluationPrompt}\n\n${presentationText}`, 'erc');
  feedbackBox.textContent = feedback;
  downloadBtn.disabled = false;
});


  downloadBtn.addEventListener('click', () => {
    const feedbackText = feedbackBox.textContent;
    const converted = '\uFEFF' + feedbackText.replace(/\n/g, '\r\n');
    const blob = new Blob([converted], { type: 'text/plain;charset=utf-8' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'ERC_Feedback.txt';
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  });

  resetBtn.addEventListener('click', () => {
    presentationText = '';
    historyBox.innerHTML = '';
    feedbackBox.textContent = '';
    document.getElementById('recordTimerDisplay').textContent = '';
    window.stopCountdownTimer();
    downloadBtn.disabled = true;
    audioDownloadBtn.style.display = 'none';
  });
}