import { sendToWhisper, evaluateInterviewAnswer, getFollowUpQuestion } from '../api.js';
import { setupRecording } from '../recorder.js';

let dialogue = [];
let recordTimer = null;
const MAX_RECORD_TIME = 60;

const efQuestions = [
  "What countries have you visited?",
  "What countries would you like to visit in the future--why?",
  "Would you like to live in a tropical climate?",
  "Would you like to live in a country or region with a lot of cold weather and snow-why or why not?",
  "Have you ever experienced dry weather, such as living in the desert?",
  "If you could live in any part of the world, where would you choose and why?",
  "If you could live in any part of South Korea, where would you choose and why?",
  "What languages do you speak other than Korean and English?",
  "What other language(s) would you like to learn?",
  "What opportunities do you have to speak English?",
  "Did you grow up in a family that supports second language learning-why or why not?",
  "How does learning other languages bridge our understanding of other cultures?",
  "Talk about a behavior within a culture that you do not understand. Why is the behavior strange to you?"
];

function getRandomEFQuestion() {
  return efQuestions[Math.floor(Math.random() * efQuestions.length)];
}

function updateHistoryUI() {
  const historyBox = document.getElementById('efHistoryText');
  if (!historyBox) return;
  historyBox.innerHTML = dialogue.map(d => `
    <div class="bubble ${d.role === 'assistant' ? 'gpt' : 'user'}">
      <span>${d.content}</span>
    </div>
  `).join('');
}

export function renderEFView() {
  const efContainer = document.getElementById('efContent');
  efContainer.innerHTML = `
    <style>
      .bubble {
        max-width: 80%;
        padding: 10px;
        margin: 6px 0;
        border-radius: 12px;
        font-size: 14px;
        line-height: 1.4;
        display: inline-block;
        clear: both;
      }
      .user {
        background-color: #dcf8c6;
        float: right;
        text-align: right;
      }
      .gpt {
        background-color: #f1f0f0;
        float: left;
        text-align: left;
      }
    </style>

    <div id="efQuestionBox" style="display:none">
      <strong>GPT:</strong> <span id="efQuestionText"></span>
    </div>

    <div class="controls">
      <button id="record">🎙️ Start Recording</button>
      <button id="stop" disabled>⏹️ Stop</button>
    </div>

    <div id="recordingBox" style="margin-top: 10px;">
      <p id="recordTimerDisplay" style="font-size: 13px; color: #888;"></p>
      <audio id="audio" controls></audio>
      <a id="downloadLink" style="display:none;" download></a>
      <div class="controls-row">
        <button id="downloadAudioBtn" class="orange" style="display:none;">🎧 Download Audio</button>
      </div>
    </div>

    <button id="startBtn">💬 Start Conversation</button>
    <button id="submitAnswerBtn" disabled>📤 Submit Answer</button>
    <button id="getFeedbackBtn" disabled>✅ Get Feedback</button>
    <button id="downloadFeedbackBtn" disabled>📁 Download Feedback</button>
    <button id="resetBtn">🔄 Reset</button>

    <div id="efHistoryBox">
      <h4>🗂️ Conversation History</h4>
      <div id="efHistoryText" style="overflow-y:auto; background:#fff; padding:10px; border:1px solid #ccc; border-radius:6px; font-size:14px; min-height: 150px; max-height: 250px;"></div>
    </div>

    <div id="efFeedbackBox">
      <h4>📊 Feedback</h4>
      <pre id="efFeedbackText"></pre>
    </div>
  `;

  const questionText = document.getElementById('efQuestionText');
  const startBtn = document.getElementById('startBtn');
  const submitBtn = document.getElementById('submitAnswerBtn');
  const feedbackBtn = document.getElementById('getFeedbackBtn');
  const resetBtn = document.getElementById('resetBtn');
  const timerDisplay = document.getElementById('recordTimerDisplay');
  const downloadFeedbackBtn = document.getElementById('downloadFeedbackBtn');
  const recordBtn = document.getElementById('record');
  const stopBtn = document.getElementById('stop');
  const audioElement = document.getElementById('audio');
  const audioDownloadBtn = document.getElementById('downloadAudioBtn');
  const downloadLink = document.getElementById('downloadLink');

  setupRecording(recordBtn, stopBtn, audioElement, downloadLink, (blob) => {
    const audioURL = URL.createObjectURL(blob);
    audioElement.src = audioURL;
    downloadLink.href = audioURL;
    downloadLink.setAttribute('download', 'EF_Audio.webm'); // ✅ 이 줄 추가
    audioDownloadBtn.style.display = 'inline-block';
    audioDownloadBtn.onclick = () => downloadLink.click();
  });


  recordBtn.disabled = true;

  window.startRecordingTimer = function () {
    if (recordTimer) clearInterval(recordTimer);
    let seconds = 0;
    recordTimer = setInterval(() => {
      seconds++;
      timerDisplay.textContent = `Recording: ${seconds}s`;
      if (seconds >= MAX_RECORD_TIME) {
        clearInterval(recordTimer);
        stopBtn.click();
        alert("Recording time limit reached (60s). Recording stopped automatically.");
      }
    }, 1000);
  };

  window.stopRecordingTimer = function () {
    if (recordTimer) clearInterval(recordTimer);
    timerDisplay.textContent = '';
  };

  recordBtn.addEventListener('click', () => {
    if (!questionText.textContent.trim()) {
      alert("먼저 'Start Conversation' 버튼을 눌러 질문을 받아주세요.");
      return;
    }
    window.startRecordingTimer();
  });

  stopBtn.addEventListener('click', stopRecordingTimer);

  startBtn.addEventListener('click', () => {
    const initial = getRandomEFQuestion();
    questionText.textContent = initial;
    document.getElementById('efQuestionBox').style.display = 'block';
    dialogue.push({ role: 'assistant', content: initial });
    submitBtn.disabled = false;
    startBtn.disabled = true;
    feedbackBtn.disabled = false;
    recordBtn.disabled = false;
    updateHistoryUI();
  });

submitBtn.addEventListener('click', async () => {
  const blob = window.latestRecordingBlob;
  if (!blob) return alert("녹음을 먼저 해주세요.");

  const transcript = await sendToWhisper(blob);

  // 🎯 필터링: 악의적 또는 GPT 조작 시도 탐지
  const bannedPhrases = [
    "prompt", "chatgpt", "gpt", "openai", "api key", "system prompt",
    "write me", "generate", "make a", "please write", "how do i", "code"
  ];
  const lowerTranscript = transcript.toLowerCase();
  const isAbusive = bannedPhrases.some(phrase => lowerTranscript.includes(phrase));

  if (isAbusive) {
    alert("⚠️ GPT 또는 시스템 관련 요청이 탐지되었습니다. 지정된 영어 질문에 자연스럽게 응답해주세요.");
    return;
  }

  dialogue.push({ role: 'user', content: transcript });

  const followup = await getFollowUpQuestion(transcript);
  dialogue.push({ role: 'assistant', content: followup });

  questionText.style.opacity = '0';
  setTimeout(() => {
    questionText.textContent = followup;
    questionText.style.opacity = '1';
  }, 300);

  updateHistoryUI();
});


  feedbackBtn.addEventListener('click', async () => {
    const fullText = dialogue.map(d => `${d.role}: ${d.content}`).join('\n');
    const prompt = `The following is a student-teacher conversation...`;
    const feedback = await evaluateInterviewAnswer(`${prompt}\n\n${fullText}`, 'ef');
    document.getElementById('efFeedbackText').textContent = feedback;
    feedbackBtn.disabled = true;
    downloadFeedbackBtn.disabled = false;
  });

  downloadFeedbackBtn.addEventListener('click', () => {
    const text = document.getElementById('efFeedbackText').textContent;
    const blob = new Blob(['\uFEFF' + text.replace(/\n/g, '\r\n')], { type: 'text/plain;charset=utf-8' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'EF_Feedback.txt';
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  });

  resetBtn.addEventListener('click', () => {
    dialogue = [];
    questionText.textContent = 'Click start to begin.';
    document.getElementById('efHistoryText').innerHTML = '';
    document.getElementById('efFeedbackText').textContent = '';
    timerDisplay.textContent = '';
    startBtn.disabled = false;
    submitBtn.disabled = true;
    feedbackBtn.disabled = true;
    downloadFeedbackBtn.disabled = true;
    stopRecordingTimer();
    recordBtn.disabled = true;
    audioDownloadBtn.style.display = 'none';
  });
}