// 🔊 Whisper 호출 함수 (음성 → 텍스트)
let userToken = localStorage.getItem('burnchat_token');
if (!userToken) {
  userToken = crypto.randomUUID();
  localStorage.setItem('burnchat_token', userToken);
}

export async function sendToWhisper(blob) {
  const formData = new FormData();
  formData.append("file", blob, "audio.webm");
  formData.append("model", "whisper-1");
  formData.append("language", "en");

  const response = await fetch("https://api.burnchat.space/api/whisper", {
    method: "POST",
    headers: {
      Authorization: `Bearer ${userToken}`
    },
    body: formData
  });

  if (response.status === 429) {
    alert("⚠️ 오늘의 음성 인식 사용 횟수를 초과했습니다. 내일 다시 시도해주세요.");
    return "";
  }

  if (!response.ok) {
    throw new Error("Whisper API 오류: " + (await response.text()));
  }

  const data = await response.json();
  return data.text;
}

// ✍️ GPT 평가 함수 (텍스트 → 피드백 + 클래스별 분기)
export async function evaluateInterviewAnswer(transcript, classKey = 'ec') {
  const promptsByClass = {
    ef: `You are an English conversation teacher.
The student participated in a multi-turn conversation. Evaluate their conversational English based on:
1. Follow-up Questions
2. Rejoinders
3. Comprehension Checks
4. Clarification Questions
5. Appropriate Interrupting
6. Bonus point for using all categories
Provide a score out of 5 points total. Clearly explain which categories were met and why. Include suggestions for improvement.
⚠️ Short or generic responses = score 0.`,

    ec: `You are a corporate interviewer. Provide feedback based on:
1. Logical Structure
2. Confidence
3. Specificity
4. Delivery
Format:
[Feedback]
- Logical Structure:
- Confidence:
- Specificity:
- Delivery:
⚠️ Short or generic responses = score 0.`,

    erc: `You are a university professor. Evaluate the student's summary based on:
1. Clarity of explanation
2. Reasoning and insight
3. Academic language
4. Structure and organization
⚠️ Short or generic responses = score 0.`,

    eap: `You are an academic English instructor. Assess the student's academic argument using:
1. Argument clarity
2. Supporting details
3. Academic vocabulary
4. Coherence and cohesion
⚠️ Short or generic responses = score 0.`
  };

  const systemPrompt = promptsByClass[classKey] || promptsByClass.ec;

  const response = await fetch("https://api.burnchat.space/api/eval", {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
          Authorization: `Bearer ${userToken}`
    },
    body: JSON.stringify({
      model: "gpt-3.5-turbo",
      messages: [
        { role: "system", content: systemPrompt },
        { role: "user", content: transcript }
      ],
      max_tokens: 500
    })
  });

  if (response.status === 429) {
    alert("⚠️ 오늘의 GPT 평가 횟수를 초과했습니다. 내일 다시 시도해주세요.");
    return "⚠️ 오늘은 평가 결과를 제공할 수 없습니다.";
  }

  if (!response.ok) {
    throw new Error("GPT 평가 API 오류: " + (await response.text()));
  }

  const data = await response.json();
  return data.choices[0].message.content;
}

// 🔁 후속 질문 생성 함수
export async function getFollowUpQuestion(prevAnswer) {
  const response = await fetch("https://api.burnchat.space/api/eval", {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
         Authorization: `Bearer ${userToken}`
    },
    body: JSON.stringify({
      model: "gpt-3.5-turbo",
      messages: [
        {
          role: "system",
          content: "You are a conversation partner. Ask a friendly follow-up question in English based on the user's previous response."
        },
        { role: "user", content: prevAnswer }
      ],
      max_tokens: 100
    })
  });

  if (response.status === 429) {
    alert("⚠️ 오늘의 follow-up 질문 생성 횟수를 초과했습니다. 내일 다시 시도해주세요.");
    return "⚠️ 오늘은 follow-up 질문을 생성할 수 없습니다.";
  }

  if (!response.ok) {
    throw new Error("Follow-up 질문 오류: " + (await response.text()));
  }

  const data = await response.json();
  return data.choices[0].message.content;
}
